#include "TM.h"

#include <fstream>
#include <iostream>

using namespace std;

void TM::DrawPoints(unsigned int color, int psize, PPC *ppc, 
	FrameBuffer *fb) {

	for (int vi = 0; vi < vertsN; vi++) {
		fb->DrawPoint3D(verts[vi], ppc, psize, color);
	}

}

void TM::Translate(V3 tv) {

	for (int vi = 0; vi < vertsN; vi++) {
		verts[vi] = verts[vi] + tv;
	}

}


V3 TM::GetCenter() {

	V3 ret(0.0f, 0.0f, 0.0f);
	for (int vi = 0; vi < vertsN; vi++) {
		ret = ret + verts[vi];
	}
	return ret / (float) vertsN;

}

void TM::RotateAboutArbitraryAxis(V3 aO, V3 ad, float theta) {

	for (int vi = 0; vi < vertsN; vi++) {
		verts[vi] = verts[vi].RotateThisPointAboutArbitraryAxis(aO, ad, theta);
	}

}


void TM::Position(V3 newCenter) {

	V3 oldCenter = GetCenter();
	Translate(newCenter - oldCenter);

}


// loading triangle mesh from a binary file, i.e., a .bin file from geometry folder
void TM::LoadBin(char *fname) {

	ifstream ifs(fname, ios::binary);
	if (ifs.fail()) {
		cerr << "INFO: cannot open file: " << fname << endl;
		return;
	}

	ifs.read((char*)&vertsN, sizeof(int));
	char yn;
	ifs.read(&yn, 1); // always xyz
	if (yn != 'y') {
		cerr << "INTERNAL ERROR: there should always be vertex xyz data" << endl;
		return;
	}
	if (verts)
		delete verts;
	verts = new V3[vertsN];

	ifs.read(&yn, 1); // cols 3 floats
	if (colors)
		delete colors;
	colors = 0;
	if (yn == 'y') {
		colors = new V3[vertsN];
	}

	ifs.read(&yn, 1); // normals 3 floats
	if (normals)
		delete []normals;
	normals = 0;
	if (yn == 'y') {
		normals = new V3[vertsN];
	}

	ifs.read(&yn, 1); // texture coordinates 2 floats
	float *tcs = 0; // don't have texture coordinates for now
	if (tcs)
		delete []tcs;
	tcs = 0;
	if (yn == 'y') {
		tcs = new float[vertsN * 2];
	}


	ifs.read((char*)verts, vertsN * 3 * sizeof(float)); // load verts

	if (colors) {
		ifs.read((char*)colors, vertsN * 3 * sizeof(float)); // load colors
	}

	if (normals)
		ifs.read((char*)normals, vertsN * 3 * sizeof(float)); // load normals

	if (tcs)
		ifs.read((char*)tcs, vertsN * 2 * sizeof(float)); // load texture coordinates

	ifs.read((char*)&trisN, sizeof(int));
	if (tris)
		delete tris;
	tris = new unsigned int[trisN * 3];
	ifs.read((char*)tris, trisN * 3 * sizeof(unsigned int)); // read tiangles

	ifs.close();

	cerr << "INFO: loaded " << vertsN << " verts, " << trisN << " tris from " << endl << "      " << fname << endl;
	cerr << "      xyz " << ((colors) ? "rgb " : "") << ((normals) ? "nxnynz " : "") << ((tcs) ? "tcstct " : "") << endl;

}



void TM::DrawWireFrame(unsigned int color, PPC *ppc, FrameBuffer *fb) {

	for (int tri = 0; tri < trisN; tri++) {
		V3 Vs[3], Cs[3];
		Vs[0] = verts[tris[tri * 3 + 0]];
		Vs[1] = verts[tris[tri * 3 + 1]];
		Vs[2] = verts[tris[tri * 3 + 2]];
		if (colors) {
			Cs[0] = colors[tris[tri * 3 + 0]];
			Cs[1] = colors[tris[tri * 3 + 1]];
			Cs[2] = colors[tris[tri * 3 + 2]];
		}
		for (int ei = 0; ei < 3; ei++) {
			if (colors)
				fb->Draw3DSegment(Cs[ei], Cs[(ei + 1) % 3], ppc, 
					Vs[ei], Vs[(ei + 1) % 3]);
			else
				fb->Draw3DSegment(color, ppc, Vs[ei], Vs[(ei + 1) % 3]);
		}
	}

}

void TM::VisualizeNormals(float nl, PPC *ppc, FrameBuffer *fb) {

	if (!normals)
		return;

	for (int vi = 0; vi < vertsN; vi++) {
		fb->Draw3DSegment(colors[vi], V3(1.0f, 0.0f, 0.0f), ppc,
			verts[vi], verts[vi] + normals[vi].Normalized() * nl);
	}

}

void TM::LightD(V3 ld, float ka) {

	for (int vi = 0; vi < vertsN; vi++) {
		colors[vi] = V3(1.0f, 0.0f, 0.0f);
		colors[vi] = colors[vi].Light(normals[vi], ld, ka);
	}

}

void TM::LightP(V3 L, float ka) {

	for (int vi = 0; vi < vertsN; vi++) {
		colors[vi] = V3(1.0f, 0.0f, 0.0f);
		V3 ld = (L - verts[vi]).Normalized();
		colors[vi] = colors[vi].Light(normals[vi], ld, ka);
	}

}

void TM::SetEEQs(M33 pvs, M33 &eeqs) {

	for (int ei = 0; ei < 3; ei++) {
		int ei1 = (ei + 1) % 3;
		float x0 = pvs[ei][0];
		float y0 = pvs[ei][1];
		float x1 = pvs[ei1][0];
		float y1 = pvs[ei1][1];
		eeqs[ei][0] = y1 - y0;
		eeqs[ei][1] = -x1 + x0;
		eeqs[ei][2] = y0*(x1 - x0) - x0*(y1 - y0);
		int ei2 = (ei + 2) % 3;
		if (eeqs[ei] * pvs[ei2] < 0.0f)
			eeqs[ei] = eeqs[ei] * -1.0f;
	}

}

void TM::RenderFilled(PPC *ppc, FrameBuffer *fb) {

	if (!projverts)
		projverts = new V3[vertsN];

	for (int vi = 0; vi < vertsN; vi++) {
		ppc->Project(verts[vi], projverts[vi]);
	}

	for (int tri = 0; tri < trisN; tri++) {
		if (projverts[tris[3 * tri + 0]][0] == FLT_MAX)
			continue;
		if (projverts[tris[3 * tri + 1]][0] == FLT_MAX)
			continue;
		if (projverts[tris[3 * tri + 2]][0] == FLT_MAX)
			continue;
		AABB aabb(projverts[tris[3 * tri + 0]]);
		aabb.AddPoint(projverts[tris[3 * tri + 1]]);
		aabb.AddPoint(projverts[tris[3 * tri + 2]]);
		int left = (int)(aabb.c0[0]+.5f);
		int right = (int)(aabb.c1[0]-.5f);
		int top = (int)(aabb.c0[1]+.5f);
		int bottom = (int)(aabb.c1[1]-.5f);
		M33 pvs;
		pvs[0] = projverts[tris[3 * tri + 0]];
		pvs[1] = projverts[tris[3 * tri + 1]];
		pvs[2] = projverts[tris[3 * tri + 2]];
		V3 pvzs = pvs.GetColumn(2);
		pvs.SetColumn(2, V3(1.0f, 1.0f, 1.0f));
		V3 zabc = pvs.Inverted()*pvzs;
		M33 colsm;
		colsm[0] = colors[tris[3 * tri + 0]];
		colsm[1] = colors[tris[3 * tri + 1]];
		colsm[2] = colors[tris[3 * tri + 2]];
		M33 rgbabc = (pvs.Inverted()*colsm).Transposed();
		M33 tcsm;
		if (tcs) {
			tcsm[0] = tcs[tris[3 * tri + 0]];
			tcsm[1] = tcs[tris[3 * tri + 1]];
			tcsm[2] = tcs[tris[3 * tri + 2]];
		}
		M33 tcsabc = (pvs.Inverted()*tcsm).Transposed();
		M33 eeqs; SetEEQs(pvs, eeqs);

		for (int v = top; v <= bottom; v++) {
			for (int u = left; u <= right; u++) {
				V3 pixcv(.5f + (float)u, .5f + (float)v, 1.0f);
				V3 sdv = eeqs*pixcv;
				if (sdv[0] < 0.0f || sdv[1] < 0.0f || sdv[2] < 0.0f)
					continue;
				float currz = zabc*pixcv;
				if (fb->IsFarther(u, v, currz))
					continue;
				V3 currcol = rgbabc * pixcv;
				if (tex) {
					V3 currtc = tcsabc * pixcv;
					int tu = (int)(currtc[0] * (float)tex->w);
					int tv = (int)(currtc[1] * (float)tex->h);
					currcol.SetColor(tex->Get(tu, tv));
				}
				fb->SetSafe(u, v, currcol.GetColor());
				fb->SetSafeZB(u, v, currz);
			}
		}
	}

}


void TM::SetQuad(V3 *vs) {

	vertsN = 4;
	trisN = 2;
	
	//  write allocate method
	verts = new V3[vertsN];
	projverts = new V3[vertsN];
	normals = new V3[vertsN];
	colors = new V3[vertsN];
	tcs = new V3[vertsN];
	tris = new unsigned int[3 * trisN];

	for (int vi = 0; vi < vertsN; vi++)
		verts[vi] = vs[vi];
	int tri = 0;
	tris[3 * tri + 0] = 0;
	tris[3 * tri + 1] = 1;
	tris[3 * tri + 2] = 2;
	tri++;
	tris[3 * tri + 0] = 2;
	tris[3 * tri + 1] = 3;
	tris[3 * tri + 2] = 0;

	tcs[0] = V3(0.0f, 0.0f, 1.0f);
	tcs[1] = V3(0.0f, 1.0f, 1.0f);
	tcs[2] = V3(1.0f, 1.0f, 1.0f);
	tcs[3] = V3(1.0f, 0.0f, 1.0f);

	tex = new FrameBuffer(10, 10, 32, 32);
	tex->SetChecker(4, 0xFF000000, 0xFFFFFFFF);

}
