#include <GL/glew.h>
#include "framebuffer.h"
#include "math.h"
#include "scene.h"

#include <tiffio.h>

using namespace std;

#include <iostream>
#include <fstream>
#include <strstream>


FrameBuffer::FrameBuffer(int u0, int v0, int _w, int _h) : 
	Fl_Gl_Window(u0, v0, _w, _h, 0) {

	w = _w;
	h = _h;
	pix = new unsigned int[w*h];
	zb = new float[w*h];
	ishw = 0;
	needInitHW = 1;
}

void FrameBuffer::draw() {

	if (!ishw)
		glDrawPixels(w, h, GL_RGBA, GL_UNSIGNED_BYTE, pix);
	else {
		if (needInitHW) {
			scene->InitHWRendering();
			needInitHW = 0;
		}
		scene->RenderHW();
	}
}

int FrameBuffer::handle(int event) {

	switch (event)
	{
	case FL_KEYBOARD: {
		KeyboardHandle();
		return 0;
	}
	case FL_MOVE: {
		int u = Fl::event_x();
		int v = Fl::event_y();
		if (u < 0 || u > w - 1 || v < 0 || v > h - 1)
			return 0;
		V3 cv; cv.SetColor(Get(u, v));
		cerr << u << " " << v << " " << "c: " << cv << "                  \r";
		return 0;
	}
	default:
		return 0;
	}
	return 0;
}

void FrameBuffer::KeyboardHandle() {

	int key = Fl::event_key();
	switch (key) {
	case FL_Left: {
		cerr << "INFO: pressed left arrow key";
		cerr << endl << endl;
		scene->Render();
		break;
	}
	case 'a': {
		cerr << "INFO: pan camera";

	}
	default:
		cerr << "INFO: do not understand keypress" << endl;
		return;
	}

}

// load a tiff image to pixel buffer
void FrameBuffer::LoadTiff(char* fname) {
	TIFF* in = TIFFOpen(fname, "r");
	if (in == NULL) {
		cerr << fname << " could not be opened" << endl;
		return;
	}

	int width, height;
	TIFFGetField(in, TIFFTAG_IMAGEWIDTH, &width);
	TIFFGetField(in, TIFFTAG_IMAGELENGTH, &height);
	if (w != width || h != height) {
		w = width;
		h = height;
		delete[] pix;
		pix = new unsigned int[w*h];
		size(w, h);
		glFlush();
		glFlush();
	}

	if (TIFFReadRGBAImage(in, w, h, pix, 0) == 0) {
		cerr << "failed to load " << fname << endl;
	}

	TIFFClose(in);
}

// save as tiff image
void FrameBuffer::SaveAsTiff(char *fname) {

	TIFF* out = TIFFOpen(fname, "w");

	if (out == NULL) {
		cerr << fname << " could not be opened" << endl;
		return;
	}

	TIFFSetField(out, TIFFTAG_IMAGEWIDTH, w);
	TIFFSetField(out, TIFFTAG_IMAGELENGTH, h);
	TIFFSetField(out, TIFFTAG_SAMPLESPERPIXEL, 4);
	TIFFSetField(out, TIFFTAG_BITSPERSAMPLE, 8);
	TIFFSetField(out, TIFFTAG_ORIENTATION, ORIENTATION_TOPLEFT);
	TIFFSetField(out, TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG);
	TIFFSetField(out, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_RGB);

	for (uint32 row = 0; row < (unsigned int)h; row++) {
		TIFFWriteScanline(out, &pix[(h - row - 1) * w], row);
	}

	TIFFClose(out);
}



void FrameBuffer::Set(unsigned int color) {

	for (int uv = 0; uv < w*h; uv++)
		pix[uv] = color;

}

void FrameBuffer::SetZB(float z0) {

	for (int uv = 0; uv < w*h; uv++)
		zb[uv] = z0;

}


unsigned int FrameBuffer::Get(int u, int v) {

	if (u < 0 || u > w - 1 || v < 0 || v > h - 1)
		return 0XFF000000;
	return pix[(h - 1 - v)*w + u];

}

float FrameBuffer::GetZB(int u, int v) {

	return zb[(h - 1 - v)*w + u];

}


void FrameBuffer::SetSafe(int u, int v,
	unsigned int color) {

	if (u < 0 || u > w - 1 || v < 0 || v > h - 1)
		return;

	Set(u, v, color);

}

void FrameBuffer::SetSafeZB(int u, int v, float z) {

	if (u < 0 || u > w - 1 || v < 0 || v > h - 1)
		return;

	SetZB(u, v, z);

}


void FrameBuffer::Set(int u, int v, unsigned int color) {

	pix[(h - 1 - v)*w + u] = color;

}

void FrameBuffer::SetZB(int u, int v, float z) {

	zb[(h - 1 - v)*w + u] = z;

}


void FrameBuffer::SetChecker(int cw, unsigned int col0,
	unsigned int col1) {

	for (int v = 0; v < h; v++) {
		for (int u = 0; u < w; u++) {
			int cu, cv;
			cu = u / cw;
			cv = v / cw;
			if ((cu+cv)%2)
				Set(u, v, col0);
			else
				Set(u, v, col1);
		}
	}


}


void FrameBuffer::DrawPoint2D(V3 P, int psize, unsigned int color) {


	int up = (int)P[0];
	int vp = (int)P[1];

	for (int v = vp - psize / 2; v <= vp + psize / 2; v++) {
		for (int u = up - psize / 2; u <= up + psize / 2; u++) {

			if (IsFarther(u, v, P[2]))
				continue;
			SetSafe(u, v, color);
			SetSafeZB(u, v, P[2]);

		}
	}


}


void FrameBuffer::DrawPoint3D(V3 P, PPC *ppc, int psize,
	unsigned int color) {

	V3 Pp;
	if (!ppc->Project(P, Pp))
		return;

	DrawPoint2D(Pp, psize, color);

}

void FrameBuffer::Draw3DSegment(unsigned int color, PPC *ppc,
	V3 V0, V3 V1) {

	V3 pV0, pV1;
	if (!ppc->Project(V0, pV0))
		return;
	if (!ppc->Project(V1, pV1))
		return;
	Draw2DSegment(color, pV0, pV1);
}

void FrameBuffer::Draw3DSegment(V3 C0, V3 C1, PPC *ppc,
	V3 V0, V3 V1) {

	V3 pV0, pV1;
	if (!ppc->Project(V0, pV0))
		return;
	if (!ppc->Project(V1, pV1))
		return;
	Draw2DSegment(C0, C1, pV0, pV1);
}


void FrameBuffer::Draw2DSegment(unsigned int color, V3 pV0, V3 pV1) {

	V3 v2d0 = pV0;
	V3 v2d1 = pV1;
	v2d0[2] = 0.0f;
	v2d1[2] = 0.0f;
	int pixn = (int)((v2d0-v2d1).Length() + 2);
	for (int si = 0; si < pixn; si++) {
		V3 currP = pV0 + (pV1 - pV0)*(float)si / (float)(pixn - 1);
		SetSafe((int) currP[0], (int) currP[1], color);
	}

}

int FrameBuffer::IsFarther(int u, int v, float z) {

	if (u < 0 || v < 0 || u > w - 1 || v > h - 1)
		return 1;
	float currz = GetZB(u, v);
	return (z < currz);

}

void FrameBuffer::Draw2DSegment(V3 C0, V3 C1, V3 pV0, V3 pV1) {

	V3 v2d0 = pV0;
	V3 v2d1 = pV1;
	v2d0[2] = 0.0f;
	v2d1[2] = 0.0f;
	int pixn = (int)((v2d0-v2d1).Length() + 2);
	for (int si = 0; si < pixn; si++) {
		V3 currP = pV0 + (pV1 - pV0)*(float)si / (float)(pixn - 1);
		V3 currC = C0 + (C1 - C0)*(float)si / (float)(pixn - 1);
		unsigned int color = currC.GetColor();
		int u = (int)currP[0];
		int v = (int)currP[1];
		if (IsFarther(u, v, currP[2]))
			continue;
		SetSafe(u, v, color);
		SetSafeZB(u, v, currP[2]);
	}

}


void FrameBuffer::DrawFBPointCloud(FrameBuffer *pcfb, PPC *pcppc, PPC *ppc) {

	for (int v = 0; v < pcfb->h; v++) {
		for (int u = 0; u < pcfb->w; u++) {
			float currz = pcfb->GetZB(u, v);
			if (currz == 0.0f)
				continue;
			V3 P = pcppc->Unproject(u, v, currz);
			DrawPoint3D(P, ppc, 1, pcfb->Get(u, v));
		}
	}


}
