/* init.c -- Initialization and completion routines for the wwbump ACE	*/

#include <stdlib.h>
#include <string.h>
#include <wwbump.h>
#include <wwbcc.h>

/* Initialization for the wwbump ACE */
ix_error ix_init(int argc, char **argv, ix_ace ** ap)
{
    struct wwbump *wwb;
    ix_error e;
    (void)argc;

    /* Set so ix_fini won't free a random value if ix_init fails */
    *ap = 0;

    /* Allocate memory for the WWBump structure (includes ix_ace) */
    wwb = malloc(sizeof(struct wwbump));
    if ( wwb == NULL )
        return ix_error_new(IX_ERROR_LEVEL_LOCAL, IX_ERROR_OOM, 0,
    			    "couldn't allocate memory for ACE");

    /* Microengine mask is always passed as the third argument */
    wwb->ue = atoi(argv[2]);

    /* Set blockname used to associate the ACE with its microblock */
    strcpy(wwb->bname, "WWBUMP");

    /* The name of the ACE is always the 2nd argument */
    /* The first argument is the name of the executable */
    wwb->name[sizeof(wwb->name) - 1] = '\0';
    strncpy(wwb->name, argv[1], sizeof(wwb->name) - 1);

    /* Initializes the ix_ace handle (including dispatch loop, control */
    /* access point, etc) */
    e = ix_ace_init(&wwb->ace, wwb->name);
    if (e) {
        free(wwb);
        return ix_error_new(0,0,e,"Error in ix_ace_init()\n");
    }

    /* Initialize a connection to the resource manager */
    e = RmInit();
    if (e) {
        ix_ace_fini(&wwb->ace);
        free(wwb);
        return ix_error_new(0,0,e,"Error in RmInit()\n");
    }

    /* Register with the resource manager (including exception handler) */
    e = RmRegister(&wwb->tag, wwb->bname,&wwb->ace, exception, wwb, 
    		   wwb->ue);
    if (e) {
        RmTerm();
        ix_ace_fini(&wwb->ace);
        free(wwb);
        return ix_error_new(0,0,e,"Error in RmRegister()\n");
    }

    /* Initialize crosscalls */
    e = cc_init(wwb);
    if ( e ) { 
        RmUnRegister(&wwb->tag);
        RmTerm();
        ix_ace_fini(&wwb->ace);
        free(wwb);
        return e;
    }

    *ap = &wwb->ace;
    return 0;
}

ix_error ix_fini(int argc, char **argv, ix_ace * ap)
{
    struct wwbump *wwb = (struct wwbump *) ap;
    ix_error e;
    (void)argc;
    (void)argv;

    /* ap == 0 if  ix_init() fails */
    if ( ! ap )
      return 0;

    /* Finalize crosscalls */
    e = cc_fini(wwb);
    if ( e )
        return e;

    /* Unregister the exception handler and microblocks */
    e = RmUnRegister(wwb->tag);
    if ( e )
        return ix_error_new(0,0,e,"Error in RmUnRegister()\n");

    /* Terminate connection with resource manager */
    e = RmTerm();
    if ( e )
        return ix_error_new(0,0,e,"Error in RmTerm()\n");

    /* Finalize the ix_ace handle */
    e = ix_ace_fini(&wwb->ace);
    if ( e )
        return ix_error_new(0,0,e,"Error in ix_ace_fini()\n");

    /* Free the malloc()ed memory */
    free(wwb);
    return 0;
}
