#include <stdio.h>
#include <sys/types.h>
#include <ip.h>
#include <ether.h>
#include <arp.h>
#include <tcp.h>
#include <udp.h>
#include <icmp.h>
#include <inet_services.h>
#include <sniff_print.h>
#include <setbyteorder.h>

int NRcvd = 0;

char outBuf[2048];

struct addrent addrtable[] = { 
{0, 0, 0, 0, 0, 0, 0, ""},
{0x800a03fa, 0x0, 0x0, 0xc, 0xb, 0x47, 0x64, "cisco1.cs.purdue.edu"},
{0x800a030a, 0x8, 0x0, 0x20, 0x1c, 0xf, 0xee, "ector.cs.purdue.edu"},
{0x800a0308, 0x8, 0x0, 0x20, 0xe, 0x37, 0xe, "gwen.cs.purdue.edu"},
{0x800a031f, 0x0, 0xc0, 0x4f, 0x79, 0xea, 0x61, "helga.cs.purdue.edu"},
{0x800a0364, 0x0, 0xc0, 0x4f, 0x79, 0xa3, 0xaf, "xinuserver.cs.purdue.edu"},
{0x800a0365, 0x0, 0xc0, 0x4f, 0x79, 0xea, 0x62, "xinu1.cs.purdue.edu"},
{0x800a0366, 0x0, 0xc0, 0x4f, 0x79, 0xed, 0xcf, "xinu2.cs.purdue.edu"},
{0x800a0367, 0x0, 0xc0, 0x4f, 0x79, 0xea, 0x6d, "xinu3.cs.purdue.edu"},
{0x800a0368, 0x0, 0xc0, 0x4f, 0x79, 0xea, 0x58, "xinu4.cs.purdue.edu"},
{0x800a0369, 0x0, 0xc0, 0x4f, 0x79, 0xea, 0x63, "xinu5.cs.purdue.edu"},
{0x800a036a, 0x0, 0xc0, 0x4f, 0x79, 0xed, 0x87, "xinu6.cs.purdue.edu"},
{0x800a036b, 0x0, 0xc0, 0x4f, 0x79, 0xed, 0xaf, "xinu7.cs.purdue.edu"},
{0x800a036c, 0x0, 0xc0, 0x4f, 0x79, 0xed, 0xa2, "xinu8.cs.purdue.edu"},
{0x800a036d, 0x0, 0xc0, 0x4f, 0x79, 0xed, 0xdb, "xinu9.cs.purdue.edu"},
{0x800a036e, 0x0, 0xc0, 0x4f, 0x79, 0xea, 0x70, "xinu10.cs.purdue.edu"},
{0x800a036f, 0x0, 0xc0, 0x4f, 0x79, 0xed, 0xd8, "xinu11.cs.purdue.edu"},
{0x800a0370, 0x0, 0xc0, 0x4f, 0x79, 0xed, 0x95, "xinu12.cs.purdue.edu"},
{0x800a0371, 0x0, 0xc0, 0x4f, 0x79, 0xed, 0xc9, "xinu13.cs.purdue.edu"},
{0x800a0372, 0x0, 0xc0, 0x4f, 0x79, 0xed, 0xdc, "xinu14.cs.purdue.edu"},
{0x800a0373, 0x0, 0xc0, 0x4f, 0x79, 0xed, 0xb3, "xinu15.cs.purdue.edu"},
{0x800a0374, 0x0, 0xc0, 0x4f, 0x79, 0xed, 0xa7, "xinu16.cs.purdue.edu"},
{0x800a0375, 0x0, 0xc0, 0x4f, 0x79, 0xea, 0x75, "xinu17.cs.purdue.edu"},
{0x800a0376, 0x0, 0xc0, 0x4f, 0x79, 0xed, 0xd6, "xinu18.cs.purdue.edu"},
{0x800a0377, 0x0, 0xc0, 0x4f, 0x79, 0xea, 0x79, "xinu19.cs.purdue.edu"},
{0x800a0378, 0x0, 0xc0, 0x4f, 0x79, 0xed, 0xde, "xinu20.cs.purdue.edu"},
{0x800a03c9, 0x0, 0xaa, 0x0, 0xa2, 0xec, 0xfa, "xinu101.cs.purdue.ed,"},
{0x800a03ca, 0x0, 0xaa, 0x0, 0xa3, 0x49, 0x93, "xinu102.cs.purdue.ed,"},
{0x800a03cb, 0x0, 0xaa, 0x0, 0xa3, 0x7a, 0x10, "xinu103.cs.purdue.edu"},
{0x800a03cc, 0x0, 0xaa, 0x0, 0xa3, 0xd3, 0x20, "xinu104.cs.purdue.edu"},
{0x800a03cd, 0x0, 0xaa, 0x0, 0xa3, 0x48, 0x3, "xinu105.cs.purdue.edu"},
{0x800a03ce, 0x0, 0xaa, 0x0, 0xa3, 0xe0, 0xd4, "xinu106.cs.purdue.edu"},
{0x800a03cf, 0x0, 0xaa, 0x0, 0xa3, 0xd5, 0x13, "xinu107.cs.purdue.edu"},
{0x800a03d0, 0x0, 0xaa, 0x0, 0xa1, 0xb5, 0x44, "xinu108.cs.purdue.edu"},
{0x800a03d1, 0x0, 0xaa, 0x0, 0xa3, 0x75, 0x8a, "xinu109.cs.purdue.edu"},
{0x800a03d2, 0x0, 0xaa, 0x0, 0xa1, 0xb4, 0x80, "xinu110.cs.purdue.edu"},
{0x800a03d3, 0x0, 0xaa, 0x0, 0xa1, 0xb2, 0x56, "xinu111.cs.purdue.edu"},
{0x800a03d4, 0x0, 0xaa, 0x0, 0xa2, 0xf1, 0x6c, "xinu112.cs.purdue.edu"},
{0x800a03d5, 0x0, 0xaa, 0x0, 0xa1, 0xa7, 0x64, "xinu113.cs.purdue.edu"},
{0x800a03d6, 0x0, 0xaa, 0x0, 0xa1, 0xa8, 0x6b, "xinu114.cs.purdue.edu"},
{0x800a03d7, 0x0, 0xaa, 0x0, 0xa3, 0x45, 0xe9, "xinu115.cs.purdue.edu"},
{0x800a03d8, 0x0, 0xaa, 0x0, 0xa3, 0xd6, 0xb4, "xinu116.cs.purdue.edu"},
{0x800a03d9, 0x0, 0xaa, 0x0, 0xa3, 0x48, 0x46, "xinu117.cs.purdue.edu"},
{0x800a03da, 0x0, 0xaa, 0x0, 0xa3, 0x75, 0xaa, "xinu118.cs.purdue.edu"},
{0x800a03db, 0x0, 0xaa, 0x0, 0xa2, 0xf1, 0xcd, "xinu119.cs.purdue.edu"},
{0x800a03dc, 0x0, 0xaa, 0x0, 0xa2, 0xef, 0x20, "xinu120.cs.purdue.edu"},
{0x800a03dd, 0x0, 0xaa, 0x0, 0xa1, 0xa7, 0x57, "xinu121.cs.purdue.edu"},
{0x800a03de, 0x0, 0xaa, 0x0, 0xa3, 0x49, 0x88, "xinu122.cs.purdue.edu"},
{0x800a03df, 0x0, 0xaa, 0x0, 0xa1, 0xb3, 0xc4, "xinu123.cs.purdue.edu"},
{0x800a03e0, 0x0, 0xaa, 0x0, 0xa3, 0x46, 0x35, "xinu124.cs.purdue.edu"},
{0x800a03e1, 0x0, 0xaa, 0x0, 0xa3, 0x45, 0xf7, "xinu125.cs.purdue.edu"},
{0x800a03e2, 0x0, 0xaa, 0x0, 0xa2, 0xf2, 0x7, "xinu126.cs.purdue.edu"},
{0x800a03e3, 0x0, 0xaa, 0x0, 0xa3, 0xe7, 0x18, "xinu127.cs.purdue.edu"},
{0x800a03e4, 0x0, 0xaa, 0x0, 0xa3, 0xe6, 0x60, "xinu128.cs.purdue.edu"},
{0x800a03e5, 0x0, 0xaa, 0x0, 0xa1, 0xb5, 0x35, "xinu129.cs.purdue.edu"},
{0x800a03e6, 0x0, 0xaa, 0x0, 0xa3, 0xaa, 0x74, "xinu130.cs.purdue.edu"},
{0x800a03e7, 0x0, 0xaa, 0x0, 0xa3, 0x49, 0xe5, "xinu131.cs.purdue.edu"},
{0x800a03e8, 0x0, 0xaa, 0x0, 0xa3, 0x49, 0xd4, "xinu132.cs.purdue.edu"},
{0x800a03ff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, "broadcast"},
{0, 0, 0, 0, 0, 0, 0, ""}
}; 


char *host_by_ip(IPaddr ipa) { 
  return addrtable[lookup_by_ip(ipa)].hostname; 
} 

char *host_by_eth(Eaddr ea) { 
  return addrtable[lookup_by_eth(ea)].hostname; 
} 

int lookup_by_ip(IPaddr ipa) { 
int i; 

  for(i = 1; addrtable[i].ipaddr && addrtable[i].ipaddr != ipa; i++); 

  if (!addrtable[i].ipaddr) return 0; 

  return i; 
} 


int lookup_by_eth(Eaddr ea) { 
int i; 

  for(i = 1; 
      addrtable[i].ipaddr && memcmp(addrtable[i].eaddr, ea, sizeof(Eaddr)); 
      i++); 

  if (!addrtable[i].ipaddr) return 0; 

  return i; 
} 


int
sprinteaddr(
   char *s, 
   Eaddr ea) 
{ 
  sprintf(s, "%02x:%02x:%02x:%02x:%02x:%02x", 
          (u_int32_t)(ea[0]), (u_int32_t)(ea[1]), (u_int32_t)(ea[2]), 
          (u_int32_t)(ea[3]), (u_int32_t)(ea[4]), (u_int32_t)(ea[5])); 
  return 0; 
} 

int 
sprintipaddr(
   char *s, 
   IPaddr ipa)
{
  unsigned char nums[4]; 
  nums[0] = ((ipa & 0xFF000000) >> 24 & 0xFF); 
  nums[1] = ((ipa & 0x00FF0000) >> 16); 
  nums[2] = ((ipa & 0x0000FF00) >> 8); 
  nums[3] = (ipa & 0xFF); 
  sprintf(s, "%d.%d.%d.%d", (u_int32_t)nums[0], (u_int32_t)nums[1], 
                            (u_int32_t)nums[2], (u_int32_t)nums[3]); 
} 


int
tcp_service(
   struct ep *pep)
{ 
  struct ip *pip; 
  struct tcp *th;

  if (!ISTCP(pep)) return -1; 

  pip = (struct ip *)pep->ep_data; 
  th = (struct tcp *)IP_DATA(pip);

  if (th->t_src <= 139) return th->t_src;
  return th->t_dst;
} 


int
udp_service(
   struct ep *pep)
{ 
  struct udp *uh;
  struct ip *pip;

  if (!ISUDP(pep)) return -1; 

  pip = (struct ip *)pep->ep_data; 
  uh = (struct udp *)IP_DATA(pip);

  if (uh->u_src <= 139) return uh->u_src;
  return uh->u_dst;
} 


char *tcp_serv(int port) { 

  switch(port) { 
    
    case    TCPS_ECHO: return "ECHO"; 
    case    TCPS_DISCARD: return "DISCARD"; 
    case    TCPS_DAYTIME: return "DAYTIME"; 
    case    TCPS_QUOTE: return "QOTD"; 
    case    TCPS_FTPDATA: return "FTP DATA"; 
    case    TCPS_FTP: return "FTP"; 
    case    TCPS_TELNET: return "TELNET"; 
    case    TCPS_SMTP: return "SMTP"; 
    case    TCPS_TIME: return "TIME";
    case    TCPS_NICNAME: return "WHOIS"; 
    case    TCPS_DOMAIN: return "DNS"; 
    case    TCPS_FINGER: return "FINGER";  
    case    TCPS_HTTP:   return "WEB"; 
    case    TCPS_SUNRPC: return "RPC"; 
    case    TCPS_NNTP: return "NNPT"; 
    case    TCPS_NETBIOSSSN: return "NETBIOS"; 
    default: return ""; 
  } 
}


char *udp_serv(int port) { 

  switch(port) { 

    case UDPS_ECHO: return "ECHO"; 
    case UDPS_DISCARD: return "DISCARD" ;
    case UDPS_DAYTIME: return "DAYTIME"; 
    case UDPS_TIME: return "TIME"; 
    case UDPS_NICNAME: return "WHOIS"; 
    case UDPS_DOMAIN: return "DNS";
    case UDPS_BOOTPS: return "BOOTPS"; 
    case UDPS_BOOTPC: return "BOOTPC"; 
    case UDPS_TFTP: return "TFTP"; 
    case UDPS_SUNRPC: return "RPC"; 
    case UDPS_NTP: return "NTP"; 
    default: return ""; 
  } 
}


char *icmp_type(int type) { 
  switch(type) { 
    case ICT_ECHORP : return "Echo Reply"; 
    case ICT_DESTUR : return "Destination Unreachable"; 
    case ICT_SRCQ : return "Source Quench"; 
    case ICT_REDIRECT : return "Redirec"; 
    case ICT_ECHORQ : return "Echo Request"; 
    case ICT_TIMEX : return "Time Exceeded for Datagram"; 
    case ICT_PARAMP : return "Paramater Problem on a Datagram"; 
    case ICT_TIMERQ : return "Timestamp Request"; 
    case ICT_TIMERP : return "Timestamp Reply"; 
    case ICT_INFORQ : return "Information Request"; 
    case ICT_INFORP : return "Information Reply"; 
    case ICT_MASKRQ : return "Address Mask Request"; 
    case ICT_MASKRP : return "Address Mask Reply"; 
    default: return "Unknown ICMP Type!"; 
  } 
} 


char *icmp_code(int type, int code) { 
  switch(type) { 
    case ICT_DESTUR : { 
      switch(code) { 
        case ICC_NETUR  : return "Network Unreachable"; 
	case ICC_HOSTUR : return "Host Unreachable"; 
	case ICC_PROTOUR: return "Protocol Unreachable"; 
	case ICC_PORTUR : return "Bad Port"; 
	case ICC_FNADF  : return "Can't Fragment";
	case ICC_SRCRT  : return "Source route failed"; 
      } 
      break; 
    } 
    case ICT_REDIRECT: { 
      if (code == ICC_NETRD) return "Network"; 
      else if (code == ICC_HOSTRD) return "Host"; 
      else if (code == IC_TOSNRD) return "Net service"; 
      else if (code == IC_TOSHRD) return "Host service"; 
      break; 
    } 
    case ICT_TIMEX: { 
      if (code == ICC_TIMEX)       return "ttl expired"; 
      else if (code == ICC_FTIMEX) return "frad time expired"; 
      break; 
    } 
  } 

  return "";
} 


int
print_verbose(
  struct ep *pep)
{ 
int look, et, type, code; 
char s[80]; 
struct arp *ah; 
struct tcp *th; 
struct ip *pip;
struct icmp *ich;
struct udp *uh; 
IPaddr ipa;
Eaddr ea;


  /* Print source address or hostname */
  if (ISIP(pep) && ((struct ip*)pep->ep_data)->ip_src != 0) { 
    if ( (look = lookup_by_ip(((struct ip *)pep->ep_data)->ip_src)))
       strcpy(s, addrtable[look].hostname);
    else
       sprintipaddr(s, ((struct ip *)pep->ep_data)->ip_src);
  } 
  else { 
    look = lookup_by_eth(pep->ep_src); 
    if (!look) 
      sprinteaddr(s, pep->ep_src);
    else
      strcpy(s, addrtable[look].hostname); 
  } 
  fprintf(stdout, "%s -> ", s); 


  /* print destindation (or broadcast) */
  if (ISBRC(pep)) { 
    fprintf(stdout, "(broadcast) "); 
  }
  else { 
    if (ISIP(pep) && ((struct ip*)pep->ep_data)->ip_dst != 0) { 
      if ( (look = lookup_by_ip(((struct ip *)pep->ep_data)->ip_src)))
         strcpy(s, addrtable[look].hostname);
      else
         sprintipaddr(s, ((struct ip *)pep->ep_data)->ip_src);
    } 
    else { 
      look = lookup_by_eth(pep->ep_src); 
      if (!look) 
        sprinteaddr(s, pep->ep_src);
      else
        strcpy(s, addrtable[look].hostname); 
    } 
    fprintf(stdout, "%s ", s); 
  } 


  et = ETYPE(pep);

  /* Handle ARP */
  if (et == EPT_ARP) { 
    ah = (struct arp *)pep->ep_data; 
    if (ah->ar_op == AR_REQUEST) { 
      fprintf(stdout, "ARP Who is "); 
      memcpy(&ipa, TPA(ah), IP_ALEN);
      sprintipaddr(s, ipa);
      fprintf(stdout, "%s\n", s); 
    } 
    if (ah->ar_op == AR_REPLY) {  
      fprintf(stdout, "ARP Response "); 
      memcpy(&ea, SHA(ah), EP_ALEN);
      sprinteaddr(s, ea);
      fprintf(stdout, "%s\n", s); 
    } 
  } /* ARP */

  /* Handle RARP */
  else if (et == EPT_RARP) { 
    ah = (struct arp *)pep->ep_data; 
    if (ah->ar_op == RA_REQUEST) { 
      fprintf(stdout, "RARP Who is "); 
      memcpy(&ea, THA(ah), EP_ALEN);
      sprinteaddr(s, ea);
      fprintf(stdout, "%s\n", s); 
    } 
    if (ah->ar_op == RA_REPLY) {  
      fprintf(stdout, "RARP Response "); 
      memcpy(&ipa, SPA(ah), IP_ALEN);
      sprintipaddr(s, ipa);
      fprintf(stdout, "%s\n", s); 
    } 

  } /* RARP */

  /* Handle IP packets! */
  else if (et == EPT_IP) { 

    pip = (struct ip *)pep->ep_data; 

    switch(IPPROTO(pep)) { 

      case IPT_ICMP : { 
        fprintf(stdout, "ICMP "); 
	ich = (struct icmp *)IP_DATA(pip);
	type = ich->ic_type; 
	fprintf(stdout, "%s ", icmp_type(type)); 
	if ((type == ICT_DESTUR) || (type == ICT_REDIRECT) || 
	    (type == ICT_TIMEX)) 
          fprintf(stdout, "(%s)\n", icmp_code(type, ich->ic_code));
        else
	  fprintf(stdout, "\n"); 
        break; 
      } /* ICMP */

      case IPT_TCP : { 
        fprintf(stdout, "TCP "); 
	th = (struct tcp *)IP_DATA(pip);
        fprintf(stdout, "%s ", tcp_serv(tcp_service(pep))); 
	fprintf(stdout, "sourceport=%d  destport=%d\n", th->t_src, th->t_dst); 
        break; 
      }  /* TCP */

      case IPT_UDP : { 
        fprintf(stdout, "UDP "); 
	uh = (struct udp *)(IP_DATA(pip)); 
        fprintf(stdout, "%s ", udp_serv(udp_service(pep))); 
	fprintf(stdout, "sourceport=%d  destport=%d\n", th->t_src, th->t_dst); 
        break; 
      }  /* UDP */

    } 

  } /* IP protocols */ 

} 


/*-----------------------------------------------------------*/

#define BIT(a) ((a) ? 1 : 0)

int
print_xverbose(
  struct ep *pep,
  int len,
  int ifn)
{ 
  NRcvd++;
  fprintf(stdout, "PACKET:  ----- On Interface %3d -----\n", ifn); 
  print_eh(pep, len); 

  len -= EP_HLEN; 

  switch(pep->ep_type) { 
    case  EPT_ARP :
    case  EPT_RARP : 
      print_arp((struct arp *)pep->ep_data); 
      break; 

    case  EPT_IP: 
      print_ip((struct ip *)pep->ep_data); 
      break; 
    default: 
      print_data((unsigned char *)pep->ep_data, len); 
  } 

  return 0; 
} 


char *etype(unsigned int type) { 
  switch(type) { 
    case EPT_LOOP : return "Loopback"; 
    case EPT_PUP  : return "Xerox PUP"; 
    case EPT_IP   : return "IP"; 
    case EPT_ARP  : return "ARP"; 
    case EPT_RARP : return "RARP"; 
    default       : return ""; 
  } 
} 

int print_eh(
   struct ep *pep,
   int len)
{
  fprintf(stdout, "ETHER:  ----- Ether Header -----\n"); 
  fprintf(stdout, "ETHER:  \n"); 
  fprintf(stdout, "ETHER:  Packet %d\n", NRcvd); 
  fprintf(stdout, "ETHER:  Packet size = %d bytes\n", len); 
  sprinteaddr(outBuf, pep->ep_dst); 
  fprintf(stdout, "ETHER:  Destination = %s (%s)\n", outBuf, host_by_eth(pep->ep_dst)); 
  sprinteaddr(outBuf, pep->ep_src); 
  fprintf(stdout, "ETHER:  Source      = %s (%s)\n", outBuf, host_by_eth(pep->ep_src)); 
  fprintf(stdout, "ETHER:  Ethertype   = %4x (%s)\n", pep->ep_type, 
          etype(pep->ep_type)); 
  fprintf(stdout, "ETHER:  \n"); 
  return 0; 
} 


char *arp_code(short opcode) { 
   switch(opcode) { 
     case AR_REQUEST : return "ARP Request"; 
     case AR_REPLY   : return "ARP Reply"; 
     case RA_REQUEST : return "RARP Request"; 
     case RA_REPLY   : return "RARP Reply"; 
     default : return "Unknown code\n"; 
   } 
} 


int print_arp(
   struct arp *pa) { 
IPaddr ipa;
Eaddr ea;

  fprintf(stdout, "ARP:  ----- ARP/RARP Frame -----\n"); 
  fprintf(stdout, "ARP:  \n"); 
  fprintf(stdout, "ARP:  Hardware type = %x\n", (u_int32_t)pa->ar_hwtype); 
  fprintf(stdout, "ARP:  Protocol Type = %x (%s)\n", (u_int32_t)pa->ar_prtype,
          (pa->ar_prtype == 0x0800) ? "IP" : "UNKNOWN");
  fprintf(stdout, "ARP:  Length of hardware address = %d bytes\n", 
          (u_int32_t)pa->ar_hwlen); 
  fprintf(stdout, "ARP:  Length of protocol address = %d bytes\n", 
          (u_int32_t)pa->ar_prlen); 
  fprintf(stdout, "ARP:  Opcode %d (%s)\n", (u_int32_t)pa->ar_op, 
          arp_code(pa->ar_op));

  if (pa->ar_hwtype == AR_HARDWARE) { 
    memcpy(&ea, SHA(pa), EP_ALEN);
    sprinteaddr(outBuf, ea); 
    fprintf(stdout, "ARP:  Sender's hardware address = %s\n", outBuf); 
  } 

  if (pa->ar_prtype == 0x0800) { 
    memcpy(&ipa, SPA(pa), sizeof(ipa));
    if (pa->ar_op != RA_REQUEST)
      sprintipaddr(outBuf, ipa); 
    else
      sprintf(outBuf, "?"); 
    fprintf(stdout, "ARP:  Sender's protocol address = %s, %s\n", outBuf,
            host_by_ip(ipa)); 
  } 

  if (pa->ar_hwtype == AR_HARDWARE) { 
    memcpy(&ea, THA(pa), EP_ALEN);
    if (pa->ar_op != AR_REQUEST)
      sprinteaddr(outBuf, ea); 
    else
      sprintf(outBuf, "?"); 
    fprintf(stdout, "ARP:  Target hardware address = %s\n", outBuf); 
  } 

  if (pa->ar_prtype == 0x0800) { 
    memcpy(&ipa, SPA(pa), sizeof(ipa));
    sprintipaddr(outBuf, ipa);
    fprintf(stdout, "ARP:  Target protocol address = %s, %s\n", outBuf,
            host_by_ip(ipa)); 
  } 
  return 0; 
} 


int
print_ip(
  struct ip *pip)
{ 
int len; 

  print_iph(pip); 

  len = pip->ip_len; 

  switch(pip->ip_proto) { 
     case IPT_ICMP :  print_icmp((struct icmp *)IP_DATA(pip), 
                                len - IP_HLEN(pip)); break; 
     case IPT_TCP  :  print_tcp((struct tcp *)IP_DATA(pip), 
                                len - IP_HLEN(pip));  break; 
     case IPT_UDP  :  print_udp((struct udp *)IP_DATA(pip), 
                                len - IP_HLEN(pip));  break; 
     default : 
       print_data(IP_DATA(pip), pip->ip_len-IP_HLEN(pip));
  } 

  return 0; 
} 

char *iph_prec(unsigned char tos) { 
  switch(tos & 0xe0) { 
    case IPP_NETCTL :  return "Network control"; 
    case IPP_INCTL  :  return "Internet control"; 
    case IPP_CRIT   :  return "Critical"; 
    case IPP_FLASHO :  return "Flash over-ride"; 
    case IPP_FLASH  :  return "Flash"; 
    case IPP_IMMED  :  return "Immediate"; 
    case IPP_PRIO   :  return "Priority"; 
    case IPP_NORMAL :  return "Normal"; 
    default         : return "";
  } 
} 

char *iph_proto(unsigned char proto) { 
  switch(proto) { 
    case IPT_ICMP : return "ICMP";
    case IPT_IGMP : return "IGMP";
    case IPT_TCP  : return "TCP";
    case IPT_EGP  : return "EGP";
    case IPT_UDP  : return "UDP";
    case IPT_OSPF : return "OSPF";
    default       : return "";
  } 
} 

int print_iph(
   struct ip *pip) {  

  fprintf(stdout, "IP:   ----- IP Header -----\n"); 
  fprintf(stdout, "IP:   \n"); 
  fprintf(stdout, "IP:   Version = %u, header length = %u bytes\n",
          (u_int32_t)(pip->ip_verlen >> 4), (u_int32_t)IP_HLEN(pip)); 
  fprintf(stdout, "IP:   Type of service = %2x\n", pip->ip_tos);
  fprintf(stdout, "IP:         %d%d%d. .... = %s\n", BIT(pip->ip_tos & 0x80), 
          BIT(pip->ip_tos &0x40),BIT(pip->ip_tos & 0x20),iph_prec(pip->ip_tos));
  fprintf(stdout, "IP:         ...%d .... = %s\n", BIT(pip->ip_tos & 0x10), 
          (pip->ip_tos & 0x10) ? "Low delay" : "Normal delay"); 
  fprintf(stdout, "IP:         .... %d... = %s\n", BIT(pip->ip_tos & 0x8), 
          (pip->ip_tos & 0x8) ? "High throughput" : "Normal delay"); 
  fprintf(stdout, "IP:         .... .%d.. = %s\n", BIT(pip->ip_tos & 0x4), 
          (pip->ip_tos & 0x8) ? "High reliability" : "Normal reliability"); 
  fprintf(stdout, "IP:   Total length = %u bytes\n", (u_int32_t)pip->ip_len); 
  fprintf(stdout, "IP:   Identification %u\n", (u_int32_t)pip->ip_id); 
  fprintf(stdout, "IP:   Flags = 0x%x\n", 
          (u_int32_t)(pip->ip_fragoff & 0xe000) >> 13); 
  fprintf(stdout, "IP:   .%d.. .... = %s\n", BIT(pip->ip_fragoff & IP_DF), 
          (pip->ip_fragoff & IP_DF) ? "may not fragment" : "may fragment"); 
  fprintf(stdout, "IP:   ..%d. .... = %s\n", BIT(pip->ip_fragoff & IP_MF), 
          (pip->ip_fragoff & IP_MF) ? "last fragment" : "more fragments"); 
  fprintf(stdout, "IP:   Fragment offset = %u bytes\n", 
          (u_int32_t)((pip->ip_fragoff & 0x1fff) << 3)); 
  fprintf(stdout, "IP:   Time to live = %u seconds/hops\n", 
          (u_int32_t)pip->ip_ttl); 
  fprintf(stdout, "IP:   Protocol = %u (%s)\n", (u_int32_t)pip->ip_proto,  
          iph_proto(pip->ip_proto));
  sprintipaddr(outBuf, pip->ip_src); 
  fprintf(stdout, "IP:   Source address = %s, %s\n", outBuf, 
          host_by_ip(pip->ip_src)); 
  sprintipaddr(outBuf, pip->ip_dst); 
  fprintf(stdout, "IP:   Destination address = %s, %s\n", outBuf, 
          host_by_ip(pip->ip_dst)); 
  return 0; 
} 

int print_icmph(
   struct icmp *ich)
{ 
  fprintf(stdout, "ICMP:  ----- ICMP Header -----\n"); 
  fprintf(stdout, "ICMP:  \n");  
  fprintf(stdout, "ICMP:  Type = %d (%s)\n", (u_int32_t)ich->ic_type, 
          icmp_type(ich->ic_type)); 
  fprintf(stdout, "ICMP:  Code = %d (%s)\n", (u_int32_t)ich->ic_code, 
          icmp_code(ich->ic_type, ich->ic_code)); 
  fprintf(stdout, "ICMP:  Checksum = %0X\n", (u_int32_t)ich->ic_cksum); 
  fprintf(stdout, "ICMP:  \n"); 
  return 0; 
} 

int print_icmp(
   struct icmp *ich, 
   int len) 
{ 
  print_icmph(ich);
  print_data(ich->ic_data, len - ICMP_HSIZ); 
  return 0;
} 


int 
print_tcp_options(
   struct tcp *th)
{
unsigned char *p, *lim, *len; 
int hlen; 
u_int32_t t1, t2; 
u_int16_t tshort; 

  if ( (hlen = TCP_HLEN(th)) == TCP_HSIZ)
  {
    fprintf(stdout, "TCP:  No Options\n");
    return 0;
  }

  fprintf(stdout, "TCP:  TCP Options : \n"); 

  p = th->t_data; 
  lim = p + (hlen - TCP_HSIZ); 
  while(p < lim) { 

    if (*p == TCPO_EOP) break; 
    if (*p == TCPO_NOP) { 
      fprintf(stdout, "TCP:  Nop option\n"); 
      p++;  
      continue; 
    } 

    switch(*p) { 
      case TCPO_MSS : 
        memcpy(&tshort, p + 2, sizeof(short)); 
        tshort = N2HS(tshort); 
        fprintf(stdout, "TCP:  Maximum Segment Size : %d\n", tshort); 
        break; 
      case TCPO_WSF : 
        fprintf(stdout, "TCP:  Window Scale Factor : %d\n", 1 << *(p + 2)); 
        break; 
      case TCPO_TIM : 
        memcpy(&t1, p + 2, sizeof(u_int32_t));
        t1 = N2HS(t1);  
        memcpy(&t2, p + 6, sizeof(u_int32_t));
        t2 = N2HS(t2); 
        fprintf(stdout, "TCP:  Timestamp = %x,  Echo Reply = %x\n", t1, t2); 
        break;
      default : 
        fprintf(stdout, "TCP:  Unknown option : %d (len = %d)\n", 
                (int)*p, (int)(*p + 1)); 
    } 

    len = p + 1; 
    p += *len; 
  } 

  return 0; 
}


int print_tcph(struct tcp *th) 
{ 
  fprintf(stdout, "TCP:  ----- TCP Header -----\n"); 
  fprintf(stdout, "TCP:  \n"); 
  fprintf(stdout, "TCP:  Source Port = %u (%s)\n", (u_int32_t)th->t_src, 
          tcp_serv(th->t_src)); 
  fprintf(stdout, "TCP:  Destination Port = %u (%s)\n", (u_int32_t)th->t_dst, 
          tcp_serv(th->t_dst)); 
  fprintf(stdout, "TCP:  Sequence number = %u\n", th->t_seq); 
  fprintf(stdout, "TCP:  Acknowledgement number = %u\n", th->t_ack); 
  fprintf(stdout, "TCP:  Data offset = %d bytes\n", (u_int32_t)TCP_HLEN(th));
  fprintf(stdout, "TCP:  Flags = 0x%u\n", (u_int32_t)th->t_code & 0x3f); 
  fprintf(stdout, "TCP:        ..%d. .... = %s\n", BIT(th->t_code & TCPF_URG), 
          (th->t_code & TCPF_URG) ? "Urgent data": "No urgent data"); 
  fprintf(stdout, "TCP:        ...%d .... = %s\n", BIT(th->t_code & TCPF_ACK), 
          (th->t_code & TCPF_ACK) ? "Acknowlegement": "non Acknowlegement"); 
  fprintf(stdout, "TCP:        .... %d... = %s\n", BIT(th->t_code & TCPF_PSH), 
          (th->t_code & TCPF_PSH) ? "Push": "No push"); 
  fprintf(stdout, "TCP:        .... .%d.. = %s\n", BIT(th->t_code & TCPF_RST), 
          (th->t_code & TCPF_RST) ? "Reset": "No Reset"); 
  fprintf(stdout, "TCP:        .... ..%d. = %s\n", BIT(th->t_code & TCPF_SYN), 
          (th->t_code & TCPF_SYN) ? "Syn": "No Syn"); 
  fprintf(stdout, "TCP:        .... ...%d = %s\n", BIT(th->t_code & TCPF_FIN), 
          (th->t_code & TCPF_FIN) ? "Fin": "No Fin"); 
  fprintf(stdout, "TCP:  Window = %u\n", (u_int32_t)th->t_window); 
  fprintf(stdout, "TCP:  Checksum = %u\n", (u_int32_t)th->t_cksum); 
  fprintf(stdout, "TCP:  Urgent Pointer = %u\n", (u_int32_t)th->t_urg); 
  print_tcp_options(th);
  fprintf(stdout, "TCP:  \n"); 
  return 0;
} 

int print_tcp(
   struct tcp *th,
   int len) 
{ 
  print_tcph(th);
  print_data((caddr_t)TCP_DATA(th), len  - TCP_HLEN(th)); 
  return 0; 
}


int print_udph(
   struct udp *uh) 
{ 
  fprintf(stdout, "UDP:  ----- UDP Header -----\n"); 
  fprintf(stdout, "UDP:  \n"); 
  fprintf(stdout, "UDP:  Source Port = %u (%s)\n", (u_int32_t)uh->u_src,
          udp_serv(uh->u_src)); 
  fprintf(stdout, "UDP:  Destination Port = %u (%s)\n", (u_int32_t)uh->u_src,
          udp_serv(uh->u_dst)); 
  fprintf(stdout, "UDP:  Length = %u\n", (u_int32_t)uh->u_len); 
  fprintf(stdout, "UDP:  Checksum = %x\n", (u_int32_t)uh->u_len); 
   
  return 0;
} 

int print_udp(
   struct udp *uh, 
   int len) 
{ 
  print_udph(uh);
  print_data(uh->u_data, len - UDP_HSIZ); 
  return 0;
} 


int print_data(
   caddr_t base, 
   int len)
{ 
   int tlen, i; 
   u_int32_t *p, off = 0; 
   unsigned char c, *cp, *lim; 

/* 
 * DATA:  ADDR:       HEX:                 ASCII
 * DATA:  00000000    00000000 00000000    ........
 * DATA:  
 */

   if (len > MAXPRLEN) len = MAXPRLEN;

   fprintf(stdout, "DATA:  ----- Data Section -----\n"); 
   fprintf(stdout, "DATA:  \n"); 
   fprintf(stdout, "DATA:  ADDR:       HEX:                 ASCII:\n"); 

   p = (u_int32_t *)base; 
   while(len >= 8) { 

#if 0  /* only works in Big endian case */
     fprintf(stdout, "DATA:  %08x    %08x %08x    ", off, *p, *(p + 1)); 
#endif 

     fprintf(stdout, "DATA:  %08X    ", off); 
     for(cp=(unsigned char *)p,lim=(unsigned char *)(p+1); cp < lim; cp++)
       fprintf(stdout, "%02X", (unsigned int)*cp);
     fprintf(stdout, " "); 
     for(cp=(unsigned char *)(p+1),lim=(unsigned char *)(p+2); cp < lim; cp++)
       fprintf(stdout, "%02X", (unsigned int)*cp);

     fprintf(stdout, "    "); 

     for(cp=(unsigned char *)p,lim=(unsigned char *)(p+1); cp < lim; cp++) {
       c = *cp; 
       if ((c < 32) || (c >= 127)) c = '.'; 
       fprintf(stdout, "%c", c); 
     } 

     p++; 

     for(cp=(unsigned char *)p,lim=(unsigned char *)(p+1); cp < lim; cp++) {
       c = *cp; 
       if ((c < 32) || (c >= 127)) c = '.'; 
       fprintf(stdout, "%c", c); 
     } 

     p++;
     fprintf(stdout, "\n"); 
     
     len -= 8; 
     off += 8; 
   } 

   if (len > 0) { 
     fprintf(stdout, "DATA:  %08x    ", off); 

     if (len >= 4) { 
       for(cp=(unsigned char *)p,lim=(unsigned char *)(p+1); cp < lim; cp++)
         fprintf(stdout, "%02X", (unsigned int)*cp);

       cp = (unsigned char *)(p + 1); 
     } 
     else
       cp = (unsigned char *)p; 

     if (tlen = len % 4) { 
       if (tlen >= 1) { 
         c = *cp++; fprintf(stdout, "%02x", (unsigned int)c);  
       } 
       if (tlen >= 2) { 
         c = *cp++; fprintf(stdout, "%02x", (unsigned int)c);  
       } 
       if (tlen >= 3)  { 
         c = *cp++; fprintf(stdout, "%02x", (unsigned int)c);  
       } 
     } 

     for(i = 0; i < (8 - len); i++)  
       fprintf(stdout, "  "); 

     if (len <= 4)
       fprintf(stdout, " "); 

     fprintf(stdout, "    "); 
     if (len >= 4) { 
       for(cp=(unsigned char *)p,lim=(unsigned char *)(p+1); cp < lim; cp++) { 
         c = *cp; 
         if ((c < 32) || (c >= 127)) c = '.'; 
         fprintf(stdout, "%c", c); 
       } 

       p++; 
     } 

     cp = (unsigned char *)p; 
     if (tlen = len % 4) { 
       if (tlen >= 1) { 
         c = *cp++; 
         if ((c < 32) || (c >= 127)) c = '.'; 
         fprintf(stdout, "%c", c); 
       } 
       if (tlen >= 2) { 
         c = *cp++; 
         if ((c < 32) || (c >= 127)) c = '.'; 
         fprintf(stdout, "%c", c); 
       } 
       if (tlen >= 3)  { 
         c = *cp++; 
         if ((c < 32) || (c >= 127)) c = '.'; 
         fprintf(stdout, "%c", c); 
       } 
     } 
     fprintf(stdout, "\n"); 
   } 

   fprintf(stdout, "DATA:  \n"); 

   return 0;   
} 
